#!/usr/bin/env python3
"""
Market Scanner - Cron-driven SPY monitor with AUTO-EXECUTION
Runs every 15 minutes during market hours.
Fetches data, scores, logs. Burns ZERO API tokens.
EXECUTES TRADES AUTOMATICALLY when signals hit threshold.

Install: pip3 install yfinance --break-system-packages
Cron:    */15 6-13 * * 1-5  cd ~/moltbot-research && python3 market_scanner.py
         (Every 15 min, 6AM-1PM AZ, Mon-Fri)
"""

import os
import sys
import json
import subprocess
from datetime import datetime, timezone, timedelta
from pdt_tracker import can_day_trade, log_day_trade, get_pdt_status

def get_buy_date():
    buy_file = os.path.join(WORKSPACE, "last_buy_date.txt")
    if os.path.exists(buy_file):
        with open(buy_file, "r") as f:
            return f.read().strip()
    return "none"

def set_buy_date(date_str):
    buy_file = os.path.join(WORKSPACE, "last_buy_date.txt")
    with open(buy_file, "w") as f:
        f.write(date_str)

# --- Try importing yfinance ---
try:
    import yfinance as yf
except ImportError:
    print("ERROR: yfinance not installed. Run: pip3 install yfinance --break-system-packages")
    sys.exit(1)

# === CONFIGURATION ===
WORKSPACE = "/home/mm/moltbot-research"
TICKER = "SPY"
VIX_TICKER = "^VIX"
NTFY_TOPIC = "mm-moltbot-alerts"
HALT_FILE = os.path.join(WORKSPACE, "HALT")
SIGNALS_FILE = os.path.join(WORKSPACE, "signals.md")
SCANNER_LOG = os.path.join(WORKSPACE, "scanner_log.csv")
SPY_TRADER = os.path.join(WORKSPACE, "spy_trader.py")

# Arizona time (UTC-7, no DST)
AZ_OFFSET = timedelta(hours=-7)

# Market hours in Arizona time (9:30 ET = 6:30 AZ, 4:00 ET = 1:00 PM AZ)
MARKET_OPEN_HOUR = 6
MARKET_OPEN_MIN = 30
MARKET_CLOSE_HOUR = 14
MARKET_CLOSE_MIN = 15
NO_NEW_TRADES_HOUR = 13
NO_NEW_TRADES_MIN = 45

# Thresholds (from risk_controls.py)
MAX_POSITION_DOLLARS = 750
STOP_LOSS_PCT = 0.03
TAKE_PROFIT_PCT = 0.05

# Signal thresholds
SUPPORT_LEVEL = 683.80       # 50-day MA - UPDATE THIS WEEKLY
VIX_PANIC = 25.0             # VIX above this = fear
VIX_CALM = 15.0              # VIX below this = complacency
PRICE_DROP_ALERT = -0.015    # -1.5% intraday = alert
PRICE_SURGE_ALERT = 0.015    # +1.5% intraday = alert

# Signal scoring
BUY_THRESHOLD = 2
SELL_THRESHOLD = -3

# Auto-execution settings
AUTO_EXECUTE = True          # Set to False to return to alert-only mode
BUY_AMOUNT_DOLLARS = 50     # Buy $50 worth of SPY on buy signal
REVERSAL_BUY_AMOUNT = 50    # Smaller position on reversal-buy signals
REVERSAL_LOOKBACK = 12       # Check last 4 scans for washout
REVERSAL_FLOOR = -5         # Score must have hit this or worse
REVERSAL_RECOVERY = -3      # Then recover to this or better


def get_az_time():
    """Get current Arizona time."""
    utc_now = datetime.now(timezone.utc)
    return utc_now + AZ_OFFSET


def is_market_hours():
    """Check if within market hours (AZ time)."""
    now = get_az_time()
    # Monday=0, Friday=4
    if now.weekday() > 4:
        return False
    market_open = now.replace(hour=MARKET_OPEN_HOUR, minute=MARKET_OPEN_MIN, second=0)
    market_close = now.replace(hour=MARKET_CLOSE_HOUR, minute=MARKET_CLOSE_MIN, second=0)
    return market_open <= now <= market_close


def is_no_new_trades_window():
    """Check if we're in the no-new-trades window (after 12:45 PM AZ)."""
    now = get_az_time()
    cutoff = now.replace(hour=NO_NEW_TRADES_HOUR, minute=NO_NEW_TRADES_MIN, second=0)
    return now >= cutoff


def is_halted():
    """Check if HALT file exists."""
    return os.path.exists(HALT_FILE)


def has_open_position():
    """Check if we currently hold SPY shares."""
    positions_file = os.path.join(WORKSPACE, "positions.csv")
    if not os.path.exists(positions_file):
        return False, 0
    
    try:
        with open(positions_file, "r") as f:
            lines = f.readlines()
        if len(lines) < 2:
            return False, 0
        
        header = lines[0].strip().split(",")
        # Find SPY position
        spy_line = None
        for line in lines[1:]:
            if line.strip().startswith("SPY,"):
                spy_line = line.strip().split(",")
                break
        if not spy_line:
            return False, 0
        last = spy_line
        pos = dict(zip(header, last))
        
        if "shares" in pos:
            shares = float(pos["shares"])
            return shares > 0, shares
    except Exception as e:
        print(f"Position check error: {e}")
    
    return False, 0


NTFY_LOG = os.path.join(WORKSPACE, "ntfy_log.txt")

def send_ntfy(message, priority="default"):
    """Send ntfy notification and log locally."""
    # Log locally first
    try:
        with open(NTFY_LOG, "a") as f:
            f.write(f"[{datetime.now(timezone.utc) + AZ_OFFSET:%Y-%m-%d %H:%M:%S}] [scanner] {message}\n")
    except Exception as e:
        print(f"ntfy log write failed: {e}")

    # Send notification
    try:
        cmd = [
            "curl", "-s",
            "-H", f"Priority: {priority}",
            "-d", message,
            f"https://ntfy.sh/{NTFY_TOPIC}"
        ]
        subprocess.run(cmd, capture_output=True, timeout=10)
    except Exception as e:
        print(f"ntfy error: {e}")


def execute_trade(action, data, score):
    """Execute a trade via spy_trader.py"""
    az_time = get_az_time()
    timestamp = az_time.strftime("%H:%M AZ")
    
    if action == "buy":
        # Calculate shares to buy based on dollar amount and current price
        # Using dollar-based buy
        cmd = ["python3", SPY_TRADER, "buy_dollars", str(BUY_AMOUNT_DOLLARS)]
        action_desc = f"BUY ~${BUY_AMOUNT_DOLLARS} SPY"
    elif action == "sell":
        cmd = ["python3", SPY_TRADER, "sell_all"]
        action_desc = "SELL ALL"
    else:
        return False, "Unknown action"
    
    try:
        result = subprocess.run(cmd, capture_output=True, text=True, timeout=60, cwd=WORKSPACE)
        output = result.stdout + result.stderr
        
        if result.returncode == 0 and ("success" in output.lower() or "executed" in output.lower() or "sold" in output.lower()):
            msg = (f"✅ AUTO-EXECUTED: {action_desc}\n"
                   f"SPY ${data['spy_price']} | Score {score}\n"
                   f"Time: {timestamp}")
            send_ntfy(msg, priority="high")
            print(f"  >>> TRADE EXECUTED: {action_desc}")
            return True, output
        else:
            if "blocked" in output.lower():
                msg = (f"🛑 TRADE BLOCKED: {action_desc}\n"
                       f"Output: {output[:200]}")
                send_ntfy(msg, priority="default")
                print(f"  >>> TRADE BLOCKED: {output}")
            else:
                msg = (f"⚠️ TRADE FAILED: {action_desc}\n"
                       f"Output: {output[:200]}")
                send_ntfy(msg, priority="urgent")
                print(f"  >>> TRADE FAILED: {output}")
            return False, output
            
    except subprocess.TimeoutExpired:
        msg = f"⚠️ TRADE TIMEOUT: {action_desc}"
        send_ntfy(msg, priority="urgent")
        return False, "Timeout"
    except Exception as e:
        msg = f"⚠️ TRADE ERROR: {action_desc}\n{str(e)}"
        send_ntfy(msg, priority="urgent")
        return False, str(e)


def fetch_market_data():
    """Fetch SPY and VIX data from Yahoo Finance."""
    data = {}

    try:
        spy = yf.Ticker(TICKER)
        spy_info = spy.fast_info
        data["spy_price"] = round(spy_info.last_price, 2)
        data["spy_prev_close"] = round(spy_info.previous_close, 2)
        data["spy_open"] = round(spy_info.open, 2)
        data["spy_day_high"] = round(spy_info.day_high, 2)
        data["spy_day_low"] = round(spy_info.day_low, 2)
        data["spy_volume"] = spy_info.last_volume

        # Calculate intraday change
        if data["spy_prev_close"] > 0:
            data["spy_change_pct"] = round(
                (data["spy_price"] - data["spy_prev_close"]) / data["spy_prev_close"], 4
            )
        else:
            data["spy_change_pct"] = 0.0

    except Exception as e:
        print(f"SPY fetch error: {e}")
        return None

    try:
        vix = yf.Ticker(VIX_TICKER)
        vix_info = vix.fast_info
        data["vix"] = round(vix_info.last_price, 2)
        data["vix_prev_close"] = round(vix_info.previous_close, 2)
        if data["vix_prev_close"] > 0:
            data["vix_change_pct"] = round(
                (data["vix"] - data["vix_prev_close"]) / data["vix_prev_close"], 4
            )
        else:
            data["vix_change_pct"] = 0.0
    except Exception as e:
        print(f"VIX fetch error: {e}")
        data["vix"] = 0
        data["vix_change_pct"] = 0

    return data



def check_overnight_gap(current_price):
    """
    Detect overnight gap-up after washout day.
    Yesterday final score <= -4 and today price > yesterday close = buy.
    Only fires before 07:45 AZ to catch gap-ups early.
    """
    now = get_az_time()
    if now.hour > 7 or (now.hour == 7 and now.minute > 45):
        return False

    if not os.path.exists(SCANNER_LOG):
        return False

    try:
        with open(SCANNER_LOG, "r") as f:
            log_lines = f.readlines()

        data_lines = [l.strip() for l in log_lines[1:] if l.strip()]
        if not data_lines:
            return False

        today_str = now.strftime("%Y-%m-%d")
        prev_day_lines = [l for l in data_lines if not l.startswith(today_str)]
        if not prev_day_lines:
            return False

        last_prev = prev_day_lines[-1].split(",")
        if len(last_prev) >= 6:
            prev_score = int(last_prev[5])
            prev_price = float(last_prev[1])
            if prev_score <= -4 and current_price > prev_price:
                return True
    except Exception as e:
        print(f"Overnight gap check error: {e}")

    return False


def check_reversal_signal(current_score):
    """
    Detect washout-recovery pattern: score hit REVERSAL_FLOOR or worse
    in last REVERSAL_LOOKBACK scans, now recovered to REVERSAL_RECOVERY or better.
    """
    if current_score < REVERSAL_RECOVERY or current_score >= BUY_THRESHOLD:
        return False

    if not os.path.exists(SCANNER_LOG):
        return False

    try:
        with open(SCANNER_LOG, "r") as f:
            lines = f.readlines()

        data_lines = [l.strip() for l in lines[1:] if l.strip()]
        recent = data_lines[-REVERSAL_LOOKBACK:] if len(data_lines) >= REVERSAL_LOOKBACK else data_lines

        for line in recent:
            parts = line.split(",")
            if len(parts) >= 6:
                past_score = int(parts[5])
                if past_score <= REVERSAL_FLOOR:
                    return True
    except Exception as e:
        print(f"Reversal check error: {e}")

    return False


def score_signal(data):
    """
    Simple quantitative scoring. No LLM needed.
    Range: roughly -5 to +5
    """
    score = 0
    reasons = []

    price = data["spy_price"]
    change = data["spy_change_pct"]
    vix = data["vix"]

    # Price vs support
    if price > SUPPORT_LEVEL * 1.02:  # 2% above support
        score += 1
        reasons.append(f"Above support ({SUPPORT_LEVEL}) +1")
    elif price < SUPPORT_LEVEL:
        score -= 2
        reasons.append(f"BELOW support ({SUPPORT_LEVEL}) -2")

    # Intraday momentum
    if change > PRICE_SURGE_ALERT:
        score += 2
        reasons.append(f"Strong rally ({change:+.2%}) +2")
    elif change > 0.005:
        score += 1
        reasons.append(f"Mild green ({change:+.2%}) +1")
    elif change < PRICE_DROP_ALERT:
        score -= 2
        reasons.append(f"Sharp drop ({change:+.2%}) -2")
    elif change < -0.005:
        score -= 1
        reasons.append(f"Mild red ({change:+.2%}) -1")

    # VIX regime
    if vix > VIX_PANIC:
        score -= 2
        reasons.append(f"VIX panic ({vix}) -2")
    elif vix > 20:
        score -= 1
        reasons.append(f"VIX elevated ({vix}) -1")
    elif vix < VIX_CALM:
        score += 1
        reasons.append(f"VIX calm ({vix}) +1")

    # Price vs open (intraday direction)
    if price > data["spy_open"] * 1.005:
        score += 1
        reasons.append("Gaining from open +1")
    elif price < data["spy_open"] * 0.995:
        score -= 1
        reasons.append("Fading from open -1")

    return score, reasons


def check_position_alerts(data):
    """Check stop-loss / take-profit on open positions and auto-execute if needed."""
    positions_file = os.path.join(WORKSPACE, "positions.csv")
    if not os.path.exists(positions_file):
        return None

    try:
        with open(positions_file, "r") as f:
            lines = f.readlines()
        if len(lines) < 2:
            return None

        # Simple CSV parse - header + last line
        header = lines[0].strip().split(",")
        # Find SPY position
        spy_line = None
        for line in lines[1:]:
            if line.strip().startswith("SPY,"):
                spy_line = line.strip().split(",")
                break
        if not spy_line:
            return False, 0
        last = spy_line
        pos = dict(zip(header, last))

        if "avg_price" in pos and "shares" in pos:
            avg = float(pos["avg_price"])
            shares = float(pos["shares"])
            if shares == 0:
                return None

            current = data["spy_price"]
            pnl_pct = (current - avg) / avg

            if pnl_pct <= -STOP_LOSS_PCT:
                alert = f"🚨 STOP LOSS HIT: SPY at ${current} ({pnl_pct:+.2%} from ${avg})"
                if AUTO_EXECUTE:
                    success, _ = execute_trade("sell", data, 0)
                    if success:
                        return f"{alert} - AUTO-SOLD"
                return f"{alert}. SELL {shares} shares NOW."
            
            elif pnl_pct >= TAKE_PROFIT_PCT:
                alert = f"💰 TAKE PROFIT HIT: SPY at ${current} ({pnl_pct:+.2%} from ${avg})"
                if AUTO_EXECUTE:
                    success, _ = execute_trade("sell", data, 0)
                    if success:
                        return f"{alert} - AUTO-SOLD"
                return f"{alert}. Consider selling {shares} shares."

    except Exception as e:
        print(f"Position check error: {e}")

    return None


def write_signals(data, score, reasons, az_time, trade_action=None):
    """Append scan result to signals.md"""
    timestamp = az_time.strftime("%Y-%m-%d %H:%M AZ")

    if score >= BUY_THRESHOLD:
        signal = "🟢 BUY SIGNAL"
    elif score <= SELL_THRESHOLD:
        signal = "🔴 SELL SIGNAL"
    elif score >= 1:
        signal = "🟡 LEAN BULLISH"
    elif score <= -1:
        signal = "🟡 LEAN BEARISH"
    else:
        signal = "⚪ NEUTRAL"

    trade_note = ""
    if trade_action:
        trade_note = f"\n**Trade:** {trade_action}"

    entry = f"""
---
## Scanner: {timestamp}
**SPY:** ${data['spy_price']} ({data['spy_change_pct']:+.2%}) | **VIX:** {data['vix']} ({data['vix_change_pct']:+.2%})
**Score:** {score} → {signal}
**Scoring:** {' | '.join(reasons)}
**Volume:** {data['spy_volume']:,} | **Range:** ${data['spy_day_low']}-${data['spy_day_high']}{trade_note}
"""

    with open(SIGNALS_FILE, "a") as f:
        f.write(entry)


def write_log(data, score, az_time):
    """Append to CSV log for later analysis."""
    timestamp = az_time.strftime("%Y-%m-%d %H:%M")
    header = "timestamp,spy_price,spy_change_pct,vix,vix_change_pct,score,volume\n"

    if not os.path.exists(SCANNER_LOG):
        with open(SCANNER_LOG, "w") as f:
            f.write(header)

    with open(SCANNER_LOG, "a") as f:
        f.write(f"{timestamp},{data['spy_price']},{data['spy_change_pct']:.4f},"
                f"{data['vix']},{data['vix_change_pct']:.4f},{score},{data['spy_volume']}\n")


def main():
    az_time = get_az_time()
    timestamp = az_time.strftime("%H:%M AZ")

    # Pre-flight checks
    if is_halted():
        print(f"[{timestamp}] HALTED - skipping scan")
        return

    if not is_market_hours():
        # Allow one scan outside hours with --force flag
        if "--force" not in sys.argv:
            print(f"[{timestamp}] Outside market hours - skipping (use --force to override)")
            return

    # Fetch data
    print(f"[{timestamp}] Fetching market data...")
    data = fetch_market_data()
    if data is None:
        send_ntfy("⚠️ Scanner: Failed to fetch market data", priority="high")
        return

    # Score
    score, reasons = score_signal(data)
    trade_action = None

    print(f"[{timestamp}] SPY ${data['spy_price']} ({data['spy_change_pct']:+.2%}) | "
          f"VIX {data['vix']} | Score: {score}")

    # Check position alerts (stop-loss / take-profit) - this may auto-sell
    pos_alert = check_position_alerts(data)
    if pos_alert:
        send_ntfy(pos_alert, priority="urgent")
        print(f"  >>> POSITION ALERT: {pos_alert}")
        if "AUTO-SOLD" in pos_alert:
            trade_action = "STOP-LOSS/TAKE-PROFIT SELL (auto)"

    # Signal-based trading
    has_position, current_shares = has_open_position()
    in_no_trade_window = is_no_new_trades_window()

    if score >= BUY_THRESHOLD:
        position_value = current_shares * data["spy_price"] if has_position else 0
        can_add_more = position_value < (MAX_POSITION_DOLLARS - BUY_AMOUNT_DOLLARS + 1)
        
        if has_position and not can_add_more:
            msg = (f"🟢 BUY SIGNAL (score {score}) - MAX POSITION (${position_value:.0f})\n"
                   f"SPY ${data['spy_price']} | No action needed.")
            send_ntfy(msg, priority="default")
            print(f"  >>> BUY SIGNAL but already holding position")
            trade_action = f"BUY SIGNAL - already holding {current_shares} shares"
        elif in_no_trade_window:
            msg = (f"🟢 BUY SIGNAL (score {score}) - TOO LATE\n"
                   f"After 12:45 PM AZ, no new positions.\n"
                   f"SPY ${data['spy_price']}")
            send_ntfy(msg, priority="default")
            print(f"  >>> BUY SIGNAL but in no-new-trades window")
            trade_action = "BUY SIGNAL - blocked (after 12:45 PM)"
        elif AUTO_EXECUTE:
            success, output = execute_trade("buy", data, score)
            if success:
                trade_action = f"BUY EXECUTED (~${BUY_AMOUNT_DOLLARS})"
                set_buy_date(datetime.now().strftime("%Y-%m-%d"))
            else:
                trade_action = f"BUY FAILED: {output[:100]}"
        else:
            msg = (f"🟢 BUY SIGNAL (score {score})\n"
                   f"SPY ${data['spy_price']} ({data['spy_change_pct']:+.2%})\n"
                   f"VIX {data['vix']}\n"
                   f"Open Moltbot to analyze and execute.")
            send_ntfy(msg, priority="high")
            print(f"  >>> BUY SIGNAL sent (manual mode)")

    elif check_overnight_gap(data["spy_price"]):
        if has_position:
            msg = (f"\U0001f305 OVERNIGHT GAP-UP after washout - ALREADY HOLDING\n"
                   f"SPY ${data['spy_price']} | Monitoring.")
            send_ntfy(msg, priority="default")
            trade_action = "OVERNIGHT GAP - already holding"
        elif in_no_trade_window:
            msg = (f"\U0001f305 OVERNIGHT GAP-UP after washout - TOO LATE\n"
                   f"After 12:45 PM AZ.\nSPY ${data['spy_price']}")
            send_ntfy(msg, priority="default")
            trade_action = "OVERNIGHT GAP - blocked (after 12:45 PM)"
        elif AUTO_EXECUTE:
            cmd = ["python3", SPY_TRADER, "buy_dollars", str(BUY_AMOUNT_DOLLARS)]
            try:
                result = subprocess.run(cmd, capture_output=True, text=True, timeout=60, cwd=WORKSPACE)
                output = result.stdout + result.stderr
                if result.returncode == 0 and ("success" in output.lower() or "executed" in output.lower()):
                    msg = (f"\U0001f305 OVERNIGHT GAP BUY EXECUTED: ~${BUY_AMOUNT_DOLLARS}\n"
                           f"SPY ${data['spy_price']} | Washout recovery detected")
                    send_ntfy(msg, priority="high")
                    trade_action = f"OVERNIGHT GAP BUY EXECUTED (~${BUY_AMOUNT_DOLLARS})"
                    set_buy_date(datetime.now().strftime("%Y-%m-%d"))
                else:
                    trade_action = f"OVERNIGHT GAP BUY FAILED: {output[:100]}"
            except Exception as e:
                trade_action = f"OVERNIGHT GAP BUY ERROR: {str(e)}"
        else:
            msg = (f"\U0001f305 OVERNIGHT GAP-UP SIGNAL\n"
                   f"Yesterday washout, gap-up today.\nSPY ${data['spy_price']}")
            send_ntfy(msg, priority="high")
            trade_action = "OVERNIGHT GAP SIGNAL (manual mode)"

    elif check_reversal_signal(score):
        # Reversal-buy: washout detected, now recovering
        if has_position:
            msg = (f"\U0001f504 REVERSAL SIGNAL (score {score}) - ALREADY HOLDING\n"
                   f"SPY ${data['spy_price']} | Monitoring.")
            send_ntfy(msg, priority="default")
            trade_action = "REVERSAL SIGNAL - already holding"
        elif in_no_trade_window:
            msg = (f"\U0001f504 REVERSAL SIGNAL (score {score}) - TOO LATE\n"
                   f"After 12:45 PM AZ.\nSPY ${data['spy_price']}")
            send_ntfy(msg, priority="default")
            trade_action = "REVERSAL SIGNAL - blocked (after 12:45 PM)"
        elif AUTO_EXECUTE:
            shares = round(REVERSAL_BUY_AMOUNT / data["spy_price"], 2)
            cmd = ["python3", SPY_TRADER, "buy_dollars", str(BUY_AMOUNT_DOLLARS)]
            try:
                result = subprocess.run(cmd, capture_output=True, text=True, timeout=60, cwd=WORKSPACE)
                output = result.stdout + result.stderr
                if result.returncode == 0 and ("success" in output.lower() or "executed" in output.lower()):
                    msg = (f"\U0001f504 REVERSAL BUY EXECUTED: {shares} shares (~${REVERSAL_BUY_AMOUNT})\n"
                           f"SPY ${data['spy_price']} | Score {score} (was \u2264{REVERSAL_FLOOR} recently)")
                    send_ntfy(msg, priority="high")
                    trade_action = f"REVERSAL BUY EXECUTED (~${REVERSAL_BUY_AMOUNT})"
                    set_buy_date(datetime.now().strftime("%Y-%m-%d"))
                else:
                    trade_action = f"REVERSAL BUY FAILED: {output[:100]}"
            except Exception as e:
                trade_action = f"REVERSAL BUY ERROR: {str(e)}"
        else:
            msg = (f"\U0001f504 REVERSAL BUY SIGNAL (score {score})\n"
                   f"Washout detected, recovering.\nSPY ${data['spy_price']}")
            send_ntfy(msg, priority="high")
            trade_action = "REVERSAL SIGNAL - manual mode"

    elif score <= SELL_THRESHOLD:
        if not has_position:
            msg = (f"🔴 SELL SIGNAL (score {score}) - NO POSITION\n"
                   f"SPY ${data['spy_price']} | Nothing to sell.")
            send_ntfy(msg, priority="default")
            print(f"  >>> SELL SIGNAL but no position to sell")
            trade_action = "SELL SIGNAL - no position"
        elif AUTO_EXECUTE:
            today = datetime.now().strftime("%Y-%m-%d")
            bought_today = get_buy_date() == today
            if bought_today and not can_day_trade():
                msg = (f"🔴 SELL SIGNAL (score {score}) - PDT BLOCK\n"
                       f"Bought today, {get_pdt_status()}\n"
                       f"Holding overnight to avoid PDT flag.")
                send_ntfy(msg, priority="high")
                print(f"  >>> SELL SIGNAL blocked by PDT rule")
                trade_action = "SELL BLOCKED - PDT rule"
            else:
                success, output = execute_trade("sell", data, score)
                if success:
                    if bought_today:
                        log_day_trade(today)
                    set_buy_date("none")
                    trade_action = f"SELL EXECUTED ({current_shares} shares)"
                else:
                    trade_action = f"SELL FAILED: {output[:100]}"
        else:
            msg = (f"🔴 SELL SIGNAL (score {score})\n"
                   f"SPY ${data['spy_price']} ({data['spy_change_pct']:+.2%})\n"
                   f"VIX {data['vix']}\n"
                   f"Open Moltbot to analyze and execute.")
            send_ntfy(msg, priority="high")
            print(f"  >>> SELL SIGNAL sent (manual mode)")
    else:
        is_market_open = (az_time.hour == 6 and az_time.minute < 45)
        if is_market_open or az_time.minute < 15 or abs(data["spy_change_pct"]) > 0.01:
            position_note = f" | Holding {current_shares} shares" if has_position else ""
            msg = (f"📊 SPY ${data['spy_price']} ({data['spy_change_pct']:+.2%}) | "
                   f"VIX {data['vix']} | Score {score}{position_note}")
            send_ntfy(msg)

    # Log
    write_signals(data, score, reasons, az_time, trade_action)
    write_log(data, score, az_time)
    # Market open summary
    if az_time.hour == 6 and az_time.minute < 45:
        position_status = f"Holding {current_shares:.2f} shares" if has_position else "Cash"
        summary = (f"🔔 MARKET OPEN\n"
                   f"SPY ${data['spy_price']} ({data['spy_change_pct']:+.2%})\n"
                   f"VIX: {data['vix']}\n"
                   f"Position: {position_status}\n"
                   f"Score: {score}")
        send_ntfy(summary, priority="default")
        print(f"  >>> MARKET OPEN SUMMARY sent")

# End-of-day summary at market close
    if az_time.hour == 14 and az_time.minute < 15:
        position_status = f"Holding {current_shares:.2f} shares" if has_position else "Cash"
        summary = (f"🔔 MARKET CLOSE\n"
                   f"SPY ${data['spy_price']} ({data['spy_change_pct']:+.2%})\n"
                   f"Range: ${data['spy_day_low']}-${data['spy_day_high']}\n"
                   f"VIX: {data['vix']}\n"
                   f"Position: {position_status}\n"
                   f"Final Score: {score}")
        send_ntfy(summary, priority="default")
        print(f"  >>> END OF DAY SUMMARY sent")
    print(f"  Logged to signals.md and scanner_log.csv")


if __name__ == "__main__":
    main()
