#!/usr/bin/env python3
"""
Algo V2 Patcher (FIXED) - February 5, 2026
Changes:
  1. BUY_THRESHOLD 3 -> 2
  2. Add reversal-buy detection (washout then recovery)
  3. Create algo_changelog.csv

Run: python3 patch_algo_v2.py
"""

import os
import shutil
from datetime import datetime

WORKSPACE = "/home/mm/moltbot-research"
SCANNER = os.path.join(WORKSPACE, "market_scanner.py")
SIGNAL_LOGIC = os.path.join(WORKSPACE, "signal_logic.py")
CHANGELOG = os.path.join(WORKSPACE, "algo_changelog.csv")


def patch_scanner():
    with open(SCANNER, "r") as f:
        code = f.read()

    # --- Change 1: BUY_THRESHOLD 3 -> 2 ---
    code = code.replace("BUY_THRESHOLD = 3", "BUY_THRESHOLD = 2")

    # --- Change 2: Add reversal config after BUY_AMOUNT_DOLLARS ---
    old_buy_amount = 'BUY_AMOUNT_DOLLARS = 50     # Buy $250 worth of SPY on buy signal'
    new_buy_amount = '''BUY_AMOUNT_DOLLARS = 50     # Buy $50 worth of SPY on buy signal
REVERSAL_BUY_AMOUNT = 50    # Smaller position on reversal-buy signals
REVERSAL_LOOKBACK = 4       # Check last 4 scans for washout
REVERSAL_FLOOR = -5         # Score must have hit this or worse
REVERSAL_RECOVERY = -2      # Then recover to this or better'''
    code = code.replace(old_buy_amount, new_buy_amount)

    # --- Change 3: Add reversal detection function before score_signal ---
    reversal_func = '''
def check_reversal_signal(current_score):
    """
    Detect washout-recovery pattern: score hit REVERSAL_FLOOR or worse
    in last REVERSAL_LOOKBACK scans, now recovered to REVERSAL_RECOVERY or better.
    """
    if current_score > REVERSAL_RECOVERY or current_score >= BUY_THRESHOLD:
        return False

    if not os.path.exists(SCANNER_LOG):
        return False

    try:
        with open(SCANNER_LOG, "r") as f:
            lines = f.readlines()

        data_lines = [l.strip() for l in lines[1:] if l.strip()]
        recent = data_lines[-REVERSAL_LOOKBACK:] if len(data_lines) >= REVERSAL_LOOKBACK else data_lines

        for line in recent:
            parts = line.split(",")
            if len(parts) >= 6:
                past_score = int(parts[5])
                if past_score <= REVERSAL_FLOOR:
                    return True
    except Exception as e:
        print(f"Reversal check error: {e}")

    return False


'''
    code = code.replace(
        'def score_signal(data):',
        reversal_func + 'def score_signal(data):'
    )

    # --- Change 4: Wire reversal into main() ---
    # KEY FIX: Target unique string in main(), not write_signals()
    # main() has: "elif score <= SELL_THRESHOLD:\n        if not has_position:"
    # write_signals() has: "elif score <= SELL_THRESHOLD:\n        signal ="

    old_sell_in_main = '    elif score <= SELL_THRESHOLD:\n        if not has_position:'

    new_reversal_and_sell = '''    elif check_reversal_signal(score):
        # Reversal-buy: washout detected, now recovering
        if has_position:
            msg = (f"\\U0001f504 REVERSAL SIGNAL (score {score}) - ALREADY HOLDING\\n"
                   f"SPY ${data['spy_price']} | Monitoring.")
            send_ntfy(msg, priority="default")
            trade_action = "REVERSAL SIGNAL - already holding"
        elif in_no_trade_window:
            msg = (f"\\U0001f504 REVERSAL SIGNAL (score {score}) - TOO LATE\\n"
                   f"After 12:45 PM AZ.\\nSPY ${data['spy_price']}")
            send_ntfy(msg, priority="default")
            trade_action = "REVERSAL SIGNAL - blocked (after 12:45 PM)"
        elif AUTO_EXECUTE:
            shares = round(REVERSAL_BUY_AMOUNT / data["spy_price"], 2)
            cmd = ["python3", SPY_TRADER, "buy", str(shares)]
            try:
                result = subprocess.run(cmd, capture_output=True, text=True, timeout=60, cwd=WORKSPACE)
                output = result.stdout + result.stderr
                if result.returncode == 0 and ("success" in output.lower() or "executed" in output.lower()):
                    msg = (f"\\U0001f504 REVERSAL BUY EXECUTED: {shares} shares (~${REVERSAL_BUY_AMOUNT})\\n"
                           f"SPY ${data['spy_price']} | Score {score} (was \\u2264{REVERSAL_FLOOR} recently)")
                    send_ntfy(msg, priority="high")
                    trade_action = f"REVERSAL BUY EXECUTED (~${REVERSAL_BUY_AMOUNT})"
                    set_buy_date(datetime.now().strftime("%Y-%m-%d"))
                else:
                    trade_action = f"REVERSAL BUY FAILED: {output[:100]}"
            except Exception as e:
                trade_action = f"REVERSAL BUY ERROR: {str(e)}"
        else:
            msg = (f"\\U0001f504 REVERSAL BUY SIGNAL (score {score})\\n"
                   f"Washout detected, recovering.\\nSPY ${data['spy_price']}")
            send_ntfy(msg, priority="high")
            trade_action = "REVERSAL SIGNAL - manual mode"

    elif score <= SELL_THRESHOLD:
        if not has_position:'''

    if old_sell_in_main not in code:
        print("ERROR: Could not find sell block in main(). Manual edit needed.")
        return False

    code = code.replace(old_sell_in_main, new_reversal_and_sell)

    with open(SCANNER, "w") as f:
        f.write(code)

    print("✅ market_scanner.py patched (targeted main() correctly)")
    return True


def patch_signal_logic():
    with open(SIGNAL_LOGIC, "r") as f:
        code = f.read()

    code = code.replace("Score >= +3: BUY signal", "Score >= +2: BUY signal")
    code = code.replace("Score >= +3: BUY", "Score >= +2: BUY")

    with open(SIGNAL_LOGIC, "w") as f:
        f.write(code)

    print("✅ signal_logic.py updated")


def create_changelog():
    header = "date,time_az,change,old_value,new_value,reason\n"
    now = datetime.now().strftime("%Y-%m-%d,%H:%M")

    entries = [
        f"{now},BUY_THRESHOLD,3,2,+3 never hit in 3 days of declining market",
        f"{now},REVERSAL_BUY,none,enabled,detect washout (score<=-5) then recovery (score>=-2) as dip-buy at $50",
    ]

    if not os.path.exists(CHANGELOG):
        with open(CHANGELOG, "w") as f:
            f.write(header)

    with open(CHANGELOG, "a") as f:
        for entry in entries:
            f.write(entry + "\n")

    print("✅ algo_changelog.csv created/updated")


if __name__ == "__main__":
    print("=== Algo V2 Patcher (FIXED) ===")
    print()

    shutil.copy(SCANNER, SCANNER + ".v1.bak")
    shutil.copy(SIGNAL_LOGIC, SIGNAL_LOGIC + ".v1.bak")
    print("Backups created (.v1.bak)")

    if patch_scanner():
        patch_signal_logic()
        create_changelog()
        print()
        print("=== Verify with: python3 market_scanner.py --force ===")
        print("=== Rollback with: cp market_scanner.py.v1.bak market_scanner.py ===")
    else:
        print("ABORTED - restoring backup")
        shutil.copy(SCANNER + ".v1.bak", SCANNER)
