#!/usr/bin/env python3
"""
PDT (Pattern Day Trader) Rule Tracker
Tracks day trades to avoid 4+ in 5 rolling business days
"""

import os
import json
from datetime import datetime, timedelta

WORKSPACE = "/home/mm/moltbot-research"
PDT_FILE = os.path.join(WORKSPACE, "day_trades.json")
MAX_DAY_TRADES = 3
ROLLING_DAYS = 5

def load_trades():
    if not os.path.exists(PDT_FILE):
        return []
    with open(PDT_FILE, "r") as f:
        return json.load(f)

def save_trades(trades):
    with open(PDT_FILE, "w") as f:
        json.dump(trades, f, indent=2)

def log_day_trade(date_str=None):
    """Log a day trade. Call when same-day buy+sell occurs."""
    if date_str is None:
        date_str = datetime.now().strftime("%Y-%m-%d")
    trades = load_trades()
    trades.append({"date": date_str, "logged": datetime.now().isoformat()})
    save_trades(trades)

def count_recent_day_trades():
    """Count day trades in last 5 business days."""
    trades = load_trades()
    cutoff = datetime.now() - timedelta(days=7)  # 7 calendar days covers 5 business
    count = 0
    for t in trades:
        trade_date = datetime.strptime(t["date"], "%Y-%m-%d")
        if trade_date >= cutoff:
            count += 1
    return count

def can_day_trade():
    """Returns True if we can do another day trade without hitting PDT."""
    return count_recent_day_trades() < MAX_DAY_TRADES

def get_pdt_status():
    """Returns status string for notifications."""
    count = count_recent_day_trades()
    return f"Day trades: {count}/{MAX_DAY_TRADES} (5-day rolling)"
