#!/usr/bin/env python3
"""
Pre-market weather report - runs at 6:00 AM AZ
One-shot: check futures, flag overnight gap, send summary
"""

import os
import json
import subprocess
from datetime import datetime, timezone, timedelta
from urllib.request import urlopen, Request
from urllib.error import URLError

WORKSPACE = "/home/mm/moltbot-research"
SCANNER_LOG = os.path.join(WORKSPACE, "scanner_log.csv")
NTFY_TOPIC = "mm-moltbot-alerts"
AZ_OFFSET = timedelta(hours=-7)

def get_az_time():
    return datetime.now(timezone.utc) + AZ_OFFSET

def send_ntfy(msg, priority="default"):
    try:
        subprocess.run(
            ["curl", "-s", "-d", msg, "-H", f"Priority: {priority}",
             f"ntfy.sh/{NTFY_TOPIC}"],
            capture_output=True, timeout=15
        )
    except Exception:
        pass

def get_yahoo_data(ticker):
    url = f"https://query1.finance.yahoo.com/v8/finance/chart/{ticker}?interval=1d&range=2d"
    headers = {"User-Agent": "Mozilla/5.0"}
    try:
        req = Request(url, headers=headers)
        with urlopen(req, timeout=10) as resp:
            return json.loads(resp.read().decode())
    except Exception:
        return None

def get_prev_close_score():
    """Get last trading day final score from scanner log."""
    if not os.path.exists(SCANNER_LOG):
        return None, None
    try:
        with open(SCANNER_LOG, "r") as f:
            lines = f.readlines()
        data_lines = [l.strip() for l in lines[1:] if l.strip()]
        if not data_lines:
            return None, None
        last = data_lines[-1].split(",")
        return float(last[1]), int(last[5])
    except Exception:
        return None, None

def main():
    now = get_az_time()
    report_lines = [f"\U0001f324 PRE-MARKET REPORT {now.strftime('%m/%d %H:%M')} AZ"]

    # SPY data
    spy_data = get_yahoo_data("SPY")
    vix_data = get_yahoo_data("%5EVIX")

    spy_pre = None
    spy_prev_close = None
    vix_val = None

    if spy_data:
        try:
            meta = spy_data["chart"]["result"][0]["meta"]
            spy_pre = meta.get("regularMarketPrice", None)
            spy_prev_close = meta.get("chartPreviousClose", None)
        except Exception:
            pass

    if vix_data:
        try:
            meta = vix_data["chart"]["result"][0]["meta"]
            vix_val = meta.get("regularMarketPrice", None)
        except Exception:
            pass

    if spy_pre and spy_prev_close:
        change_pct = (spy_pre - spy_prev_close) / spy_prev_close * 100
        direction = "\U0001f7e2" if change_pct >= 0 else "\U0001f534"
        report_lines.append(f"{direction} SPY ${spy_pre:.2f} ({change_pct:+.2f}% vs prev close ${spy_prev_close:.2f})")
    else:
        report_lines.append("SPY: data unavailable")

    if vix_val:
        vix_emoji = "\U0001f7e2" if vix_val < 18 else ("\U0001f7e1" if vix_val < 22 else "\U0001f534")
        report_lines.append(f"{vix_emoji} VIX {vix_val:.2f}")

    # Use scanner log for accurate prev close (Yahoo's chartPreviousClose is unreliable)
    prev_price, prev_score = get_prev_close_score()
    if prev_price:
        spy_prev_close = prev_price
    if prev_score is not None:
        report_lines.append(f"Prev close score: {prev_score}")
        if prev_score <= -4 and spy_pre and prev_price and spy_pre > prev_price:
            report_lines.append("\U0001f6a8 OVERNIGHT GAP-UP after washout! Watch for auto-buy at 6:30.")
        elif prev_score <= -4:
            report_lines.append("\u26a0 Yesterday was washout. Watching for gap-up at open.")
        elif prev_score >= 3:
            report_lines.append("Momentum carrying from yesterday. Watch for continuation.")

    # Outlook
    if spy_pre and spy_prev_close:
        if change_pct > 0.5:
            report_lines.append("Outlook: Bullish open expected. +2/+3 likely early.")
        elif change_pct < -0.5:
            report_lines.append("Outlook: Bearish open expected. Stay cautious.")
        else:
            report_lines.append("Outlook: Flat open. Wait for direction.")

    msg = "\n".join(report_lines)
    send_ntfy(msg, priority="default")
    print(msg)

if __name__ == "__main__":
    main()
