#!/usr/bin/env python3
"""
SPY Trading Agent - Moltbot Integration
Hard-coded risk controls. Logs all actions.
"""

import os
import sys
import csv
import json
import subprocess
import time
from datetime import datetime, date
from dotenv import load_dotenv
import robin_stocks.robinhood as rh

# ============================================
# HARD-CODED RISK CONTROLS - DO NOT MODIFY
# ============================================
MAX_POSITION_DOLLARS = 750
MAX_DAILY_TRADES = 5
STOP_LOSS_PCT = 0.03
TAKE_PROFIT_PCT = 0.05
MAX_DAILY_LOSS = 50
TICKER = "SPY"
SHARES_ONLY = True
HALT_FILE = "/home/mm/moltbot-research/HALT"
WORKSPACE = "/home/mm/moltbot-research"

# ============================================
# FILE PATHS
# ============================================
ENV_FILE = os.path.join(WORKSPACE, ".env")
TRADE_LOG = os.path.join(WORKSPACE, "executed_trades.csv")
POSITIONS_FILE = os.path.join(WORKSPACE, "positions.csv")
PENDING_FILE = os.path.join(WORKSPACE, "pending_trades.md")
SIGNALS_FILE = os.path.join(WORKSPACE, "signals.md")
DAILY_LOG = os.path.join(WORKSPACE, "daily_pnl.csv")
REALIZED_PNL_LOG = os.path.join(WORKSPACE, "realized_pnl.csv")

# ============================================
# NOTIFICATIONS
# ============================================
NTFY_TOPIC = "mm-moltbot-alerts"

NTFY_LOG = os.path.join(WORKSPACE, "ntfy_log.txt")

def notify(message):
    """Send push notification via ntfy and log locally"""
    # Log to file first (always, even if send fails)
    try:
        with open(NTFY_LOG, "a") as f:
            f.write(f"[{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}] {message}\n")
    except Exception as e:
        print(f"ntfy log write failed: {e}")

    # Send notification
    try:
        subprocess.run(
            ["curl", "-s", "-d", message, f"ntfy.sh/{NTFY_TOPIC}"],
            capture_output=True, timeout=10
        )
    except Exception as e:
        print(f"Notification failed: {e}")

# ============================================
# HALT CHECK
# ============================================
def check_halt():
    """Check if trading is halted"""
    if os.path.exists(HALT_FILE):
        print("HALT FILE DETECTED. All trading stopped.")
        notify("HALT: Trading stopped. HALT file detected.")
        return True
    return False

# ============================================
# LOGIN
# ============================================
def login():
    """Login to Robinhood"""
    load_dotenv(ENV_FILE)
    username = os.getenv("RH_USERNAME")
    password = os.getenv("RH_PASSWORD")
    if not username or not password:
        print("ERROR: Missing RH credentials in .env")
        notify("ERROR: Missing Robinhood credentials")
        return False
    try:
        rh.login(username, password)
        print("Robinhood login successful.")
        return True
    except Exception as e:
        print(f"Login failed: {e}")
        notify(f"ERROR: Robinhood login failed - {e}")
        return False

def logout():
    """Logout from Robinhood"""
    try:
        rh.logout()
        print("Logged out.")
    except:
        pass


def poll_order_status(order_id, max_wait=15):
    """Poll Robinhood for order fill status"""
    if not order_id:
        return "unknown"
    for i in range(max_wait):
        try:
            info = rh.orders.get_stock_order_info(order_id)
            state = info.get("state", "unknown")
            if state in ("filled", "cancelled", "failed", "rejected"):
                return state
            time.sleep(1)
        except Exception:
            time.sleep(1)
    return "timeout (may still fill)"

# ============================================
# TRADE LOGGING
# ============================================
def init_trade_log():
    """Create trade log CSV if it doesn't exist"""
    if not os.path.exists(TRADE_LOG):
        with open(TRADE_LOG, "w", newline="") as f:
            writer = csv.writer(f)
            writer.writerow([
                "timestamp", "action", "ticker", "shares",
                "price", "total", "reason", "status"
            ])

def log_trade(action, ticker, shares, price, reason, status):
    """Append trade to log"""
    init_trade_log()
    with open(TRADE_LOG, "a", newline="") as f:
        writer = csv.writer(f)
        writer.writerow([
            datetime.now().isoformat(),
            action, ticker, shares,
            f"{price:.2f}", f"{shares * price:.2f}",
            reason, status
        ])

# ============================================
# DAILY TRADE COUNT & LOSS TRACKING
# ============================================

def log_realized_pnl(sell_price, buy_price, shares, ticker):
    """Log actual realized profit/loss on a closed position"""
    pnl = (sell_price - buy_price) * shares
    header = "timestamp,ticker,shares,buy_price,sell_price,realized_pnl\n"
    if not os.path.exists(REALIZED_PNL_LOG):
        with open(REALIZED_PNL_LOG, "w") as f:
            f.write(header)
    with open(REALIZED_PNL_LOG, "a") as f:
        f.write(f"{datetime.now().isoformat()},{ticker},{shares:.4f},"
                f"{buy_price:.2f},{sell_price:.2f},{pnl:.2f}\n")
    return pnl

def get_todays_trades():
    """Count trades executed today"""
    if not os.path.exists(TRADE_LOG):
        return 0
    today = date.today().isoformat()
    count = 0
    with open(TRADE_LOG, "r") as f:
        reader = csv.DictReader(f)
        for row in reader:
            if row["timestamp"].startswith(today) and row["status"] in ("filled", "executed"):
                count += 1
    return count

def get_todays_pnl():
    """Calculate today's ACTUAL realized P&L from closed positions"""
    if not os.path.exists(REALIZED_PNL_LOG):
        return 0.0
    today = date.today().isoformat()
    pnl = 0.0
    with open(REALIZED_PNL_LOG, "r") as f:
        reader = csv.DictReader(f)
        for row in reader:
            if row["timestamp"].startswith(today):
                pnl += float(row["realized_pnl"])
    return pnl

# ============================================
# POSITION TRACKING
# ============================================
def update_positions():
    """Write current positions to positions.csv"""
    try:
        holdings = rh.build_holdings()
        with open(POSITIONS_FILE, "w", newline="") as f:
            writer = csv.writer(f)
            writer.writerow([
                "ticker", "shares", "avg_cost", "current_price",
                "equity", "pnl", "pnl_pct", "updated"
            ])
            for ticker, data in holdings.items():
                writer.writerow([
                    ticker,
                    data.get("quantity", 0),
                    data.get("average_buy_price", 0),
                    data.get("price", 0),
                    data.get("equity", 0),
                    data.get("equity_change", 0),
                    data.get("percent_change", 0),
                    datetime.now().isoformat()
                ])
        print(f"Positions updated: {len(holdings)} holdings")
    except Exception as e:
        print(f"Position update failed: {e}")

def get_current_position():
    """Get current SPY position"""
    try:
        holdings = rh.build_holdings()
        if TICKER in holdings:
            return {
                "shares": float(holdings[TICKER]["quantity"]),
                "avg_cost": float(holdings[TICKER]["average_buy_price"]),
                "current_price": float(holdings[TICKER]["price"]),
                "equity": float(holdings[TICKER]["equity"])
            }
    except Exception as e:
        print(f"Error getting position: {e}")
    return None

def get_cash():
    """Get available cash"""
    try:
        profile = rh.build_user_profile()
        return float(profile.get("cash", 0))
    except:
        return 0.0

# ============================================
# RISK CHECKS
# ============================================
def can_buy(price, shares, skip_loss_check=False):
    """Check all risk controls before buying"""
    reasons = []

    # Halt check
    if check_halt():
        return False, "HALTED"

    # Daily trade limit
    if get_todays_trades() >= MAX_DAILY_TRADES:
        reasons.append(f"Daily trade limit reached ({MAX_DAILY_TRADES})")

    # Daily loss limit (skip when doing sell+rebuy sequence)
    if not skip_loss_check:
        daily_pnl = get_todays_pnl()
        if daily_pnl <= -MAX_DAILY_LOSS:
            reasons.append(f"Daily loss limit hit (${daily_pnl:.2f})")

    # Position size limit
    cost = price * shares
    if cost > MAX_POSITION_DOLLARS:
        reasons.append(f"Order ${cost:.2f} exceeds max position ${MAX_POSITION_DOLLARS}")

    # Cash check
    cash = get_cash()
    if cost > cash:
        reasons.append(f"Insufficient cash: ${cash:.2f} < ${cost:.2f}")

    # Existing position check
    pos = get_current_position()
    if pos:
        total_exposure = pos["equity"] + cost
        if total_exposure > MAX_POSITION_DOLLARS:
            reasons.append(f"Total exposure ${total_exposure:.2f} exceeds max ${MAX_POSITION_DOLLARS}")

    if reasons:
        return False, "; ".join(reasons)
    return True, "All checks passed"


def can_buy_dollars(amount, skip_loss_check=False):
    """Check all risk controls before buying by dollar amount"""
    reasons = []

    if check_halt():
        return False, "HALTED"

    if get_todays_trades() >= MAX_DAILY_TRADES:
        reasons.append(f"Daily trade limit reached ({MAX_DAILY_TRADES})")

    if not skip_loss_check:
        daily_pnl = get_todays_pnl()
        if daily_pnl <= -MAX_DAILY_LOSS:
            reasons.append(f"Daily loss limit hit (${daily_pnl:.2f})")

    if amount > MAX_POSITION_DOLLARS:
        reasons.append(f"Order ${amount:.2f} exceeds max position ${MAX_POSITION_DOLLARS}")

    cash = get_cash()
    if amount > cash:
        reasons.append(f"Insufficient cash: ${cash:.2f} < ${amount:.2f}")

    pos = get_current_position()
    if pos:
        total_exposure = pos["equity"] + amount
        if total_exposure > MAX_POSITION_DOLLARS:
            reasons.append(f"Total exposure ${total_exposure:.2f} exceeds max ${MAX_POSITION_DOLLARS}")

    if reasons:
        return False, "; ".join(reasons)
    return True, "All checks passed"

def check_stop_loss():
    """Check if current position hit stop loss"""
    pos = get_current_position()
    if not pos or pos["shares"] == 0:
        return False, None
    pnl_pct = (pos["current_price"] - pos["avg_cost"]) / pos["avg_cost"]
    if pnl_pct <= -STOP_LOSS_PCT:
        return True, f"Stop loss triggered: {pnl_pct:.2%}"
    return False, None

def check_take_profit():
    """Check if current position hit take profit"""
    pos = get_current_position()
    if not pos or pos["shares"] == 0:
        return False, None
    pnl_pct = (pos["current_price"] - pos["avg_cost"]) / pos["avg_cost"]
    if pnl_pct >= TAKE_PROFIT_PCT:
        return True, f"Take profit triggered: {pnl_pct:.2%}"
    return False, None

# ============================================
# TRADE EXECUTION
# ============================================
def buy_spy(shares, reason="Signal", skip_loss_check=False):
    """Execute buy order with risk checks"""
    price = float(rh.stocks.get_latest_price(TICKER)[0])
    can, msg = can_buy(price, shares, skip_loss_check=skip_loss_check)

    if not can:
        print(f"BUY BLOCKED: {msg}")
        log_trade("BUY", TICKER, shares, price, reason, f"blocked: {msg}")
        notify(f"BUY BLOCKED: {shares} {TICKER} @ ${price:.2f} - {msg}")
        return False

    try:
        order = rh.orders.order_buy_market(TICKER, shares)
        order_id = order.get("id", None)
        status = poll_order_status(order_id)
        print(f"BUY ORDER: {shares} {TICKER} @ ~${price:.2f} - Status: {status}")
        log_trade("BUY", TICKER, shares, price, reason, status)
        notify(f"BUY {status.upper()}: {shares} {TICKER} @ ${price:.2f} - {reason}")
        update_positions()
        return status == "filled"
    except Exception as e:
        print(f"BUY FAILED: {e}")
        log_trade("BUY", TICKER, shares, price, reason, f"failed: {e}")
        notify(f"BUY FAILED: {shares} {TICKER} - {e}")
        return False


def buy_spy_dollars(amount, reason="Signal", skip_loss_check=False):
    """Execute buy order by dollar amount (fractional shares)"""
    can, msg = can_buy_dollars(amount, skip_loss_check=skip_loss_check)

    if not can:
        print(f"BUY BLOCKED: {msg}")
        price = float(rh.stocks.get_latest_price(TICKER)[0])
        log_trade("BUY", TICKER, f"${amount}", price, reason, f"blocked: {msg}")
        notify(f"BUY BLOCKED: ${amount} {TICKER} - {msg}")
        return False

    try:
        price = float(rh.stocks.get_latest_price(TICKER)[0])
        order = rh.orders.order_buy_fractional_by_price(TICKER, amount, timeInForce='gfd')
        order_id = order.get("id", None)
        status = poll_order_status(order_id)
        est_shares = round(amount / price, 4)
        print(f"BUY ORDER: ${amount} {TICKER} (~{est_shares} shares @ ~${price:.2f}) - Status: {status}")
        log_trade("BUY", TICKER, est_shares, price, reason, status)
        notify(f"BUY {status.upper()}: ${amount} {TICKER} (~{est_shares} shares @ ${price:.2f}) - {reason}")
        update_positions()
        return status == "filled"
    except Exception as e:
        print(f"BUY FAILED: {e}")
        try:
            price = float(rh.stocks.get_latest_price(TICKER)[0])
        except:
            price = 0
        log_trade("BUY", TICKER, f"${amount}", price, reason, f"failed: {e}")
        notify(f"BUY FAILED: ${amount} {TICKER} - {e}")
        return False

def sell_spy(shares, reason="Signal"):
    """Execute sell order with realized P&L tracking"""
    if check_halt():
        return False

    price = float(rh.stocks.get_latest_price(TICKER)[0])

    pos = get_current_position()
    if not pos or pos["shares"] < shares:
        available = pos["shares"] if pos else 0
        print(f"SELL BLOCKED: Only {available} shares available")
        return False

    avg_cost = pos["avg_cost"]

    try:
        order = rh.orders.order_sell_market(TICKER, shares)
        order_id = order.get("id", None)
        status = poll_order_status(order_id)
        print(f"SELL ORDER: {shares} {TICKER} @ ~${price:.2f} - Status: {status}")
        log_trade("SELL", TICKER, shares, price, reason, status)

        # Log realized P&L
        if status == "filled":
            rpnl = log_realized_pnl(price, avg_cost, shares, TICKER)
            notify(f"SELL {status.upper()}: {shares} {TICKER} @ ${price:.2f} (P&L: ${rpnl:.2f}) - {reason}")
        else:
            notify(f"SELL {status.upper()}: {shares} {TICKER} @ ${price:.2f} - {reason}")

        update_positions()
        return status == "filled"
    except Exception as e:
        print(f"SELL FAILED: {e}")
        log_trade("SELL", TICKER, shares, price, reason, f"failed: {e}")
        notify(f"SELL FAILED: {shares} {TICKER} - {e}")
        return False


def sell_spy_dollars(amount, reason="Signal"):
    """Execute sell order by dollar amount (fractional shares) with realized P&L"""
    if check_halt():
        return False

    pos = get_current_position()
    if not pos or pos["equity"] <= 0:
        print("SELL BLOCKED: No position to sell")
        return False

    if amount > pos["equity"]:
        print(f"SELL BLOCKED: ${amount} exceeds position equity ${pos['equity']:.2f}")
        return False

    avg_cost = pos["avg_cost"]

    try:
        price = float(rh.stocks.get_latest_price(TICKER)[0])
        order = rh.orders.order_sell_fractional_by_price(TICKER, amount, timeInForce='gfd')
        order_id = order.get("id", None)
        status = poll_order_status(order_id)
        est_shares = round(amount / price, 4)
        print(f"SELL ORDER: ${amount} {TICKER} (~{est_shares} shares @ ~${price:.2f}) - Status: {status}")
        log_trade("SELL", TICKER, est_shares, price, reason, status)

        # Log realized P&L
        if status == "filled":
            rpnl = log_realized_pnl(price, avg_cost, est_shares, TICKER)
            notify(f"SELL {status.upper()}: ${amount} {TICKER} (~{est_shares} shares @ ${price:.2f}, P&L: ${rpnl:.2f}) - {reason}")
        else:
            notify(f"SELL {status.upper()}: ${amount} {TICKER} (~{est_shares} shares @ ${price:.2f}) - {reason}")

        update_positions()
        return status == "filled"
    except Exception as e:
        print(f"SELL FAILED: {e}")
        try:
            price = float(rh.stocks.get_latest_price(TICKER)[0])
        except:
            price = 0
        log_trade("SELL", TICKER, f"${amount}", price, reason, f"failed: {e}")
        notify(f"SELL FAILED: ${amount} {TICKER} - {e}")
        return False

def sell_all(reason="Manual close"):
    """Sell entire SPY position"""
    pos = get_current_position()
    if not pos or pos["shares"] == 0:
        print("No position to sell.")
        return False
    shares = pos["shares"]
    if shares != int(shares):
        # Fractional position - sell by dollar amount
        return sell_spy_dollars(pos["equity"], reason)
    return sell_spy(int(shares), reason)

# ============================================
# STATUS REPORT
# ============================================
def status_report():
    """Print and log current status"""
    cash = get_cash()
    pos = get_current_position()
    daily_trades = get_todays_trades()
    daily_pnl = get_todays_pnl()

    report = []
    report.append(f"=== SPY Trading Status - {datetime.now().strftime('%Y-%m-%d %H:%M')} ===")
    report.append(f"Cash: ${cash:.2f}")
    report.append(f"Daily trades: {daily_trades}/{MAX_DAILY_TRADES}")
    report.append(f"Daily P&L (realized): ${daily_pnl:.2f} (limit: -${MAX_DAILY_LOSS})")
    report.append(f"Halt file: {'YES - TRADING STOPPED' if os.path.exists(HALT_FILE) else 'No'}")

    if pos:
        pnl_pct = (pos["current_price"] - pos["avg_cost"]) / pos["avg_cost"] * 100
        report.append(f"\nPosition: {pos['shares']:.0f} shares @ ${pos['avg_cost']:.2f}")
        report.append(f"Current: ${pos['current_price']:.2f} | P&L: {pnl_pct:.2f}%")
        report.append(f"Equity: ${pos['equity']:.2f}")
        report.append(f"Stop loss at: {-STOP_LOSS_PCT*100:.1f}% | Take profit at: {TAKE_PROFIT_PCT*100:.1f}%")
    else:
        report.append("\nNo open position.")

    output = "\n".join(report)
    print(output)
    return output

# ============================================
# MAIN - for direct testing
# ============================================
if __name__ == "__main__":
    if not login():
        sys.exit(1)

    if len(sys.argv) > 1:
        cmd = sys.argv[1]
        if cmd == "status":
            status_report()
        elif cmd == "buy" and len(sys.argv) > 2:
            buy_spy(float(sys.argv[2]), "Manual buy")
        elif cmd == "sell" and len(sys.argv) > 2:
            sell_spy(float(sys.argv[2]), "Manual sell")
        elif cmd == "sell_all":
            sell_all("Manual close all")
        elif cmd == "positions":
            update_positions()
        elif cmd == "halt":
            open(HALT_FILE, "w").close()
            print("HALT file created. Trading stopped.")
            notify("HALT: Trading manually stopped.")
        elif cmd == "resume":
            if os.path.exists(HALT_FILE):
                os.remove(HALT_FILE)
                print("HALT file removed. Trading resumed.")
                notify("RESUMED: Trading resumed.")
            else:
                print("No HALT file found.")
        elif cmd == "buy_dollars" and len(sys.argv) > 2:
            buy_spy_dollars(float(sys.argv[2]), "Manual buy (dollars)")
        elif cmd == "sell_dollars" and len(sys.argv) > 2:
            sell_spy_dollars(float(sys.argv[2]), "Manual sell (dollars)")
        else:
            print("Commands: status, buy <shares>, sell <shares>, buy_dollars <amount>, sell_dollars <amount>, sell_all, positions, halt, resume")
    else:
        status_report()

    logout()
